import filestore1.ThreeDFile;

public class Scanner{

	private double startx;
	private double starty;
	private double radius;
	private double step;
	private double gradientThresholdMax;
	private double gradientThresholdMin;
	private static final int SAMPLESIZE = 100;
	
	/** Constructor takes the coordinates of the start point, the radius
	that it is to scan out to, and the size of the sample box that it will
	use to scan the space with.
	
	@param x The X-coordinate of the scan start point
	@param y The Y-coordinate of the scan start point
	@param radius Limit the scan to an area with this radius
	@param step The spacing between sample points
	*/
	public Scanner(double x, double y, double radius, double step)
	{
		this.startx = x;
		this.starty = y;
		this.radius = radius;
		this.step = step;
	}
	
	public void setThreshold(double min, double max)
	{
		gradientThresholdMin = min;
		gradientThresholdMax = max;
	}
	
	/** This method performs most of the work. It scans the surface in
	concentric squares of increasing size, starting at (startx, starty), 
	until the square is 2*radius a side. Each sample is consists of at most 
	SAMPLESIZE points.
	*/
	public int scan()
	{
		double originX = startx - 0.5*step;
		double originY = starty - 0.5*step;
		int iterationLimit = (int)(radius / step); // Number of concentric squares to scan
		int iterations = 0;
		ThreeDFile scanset;
		int targets = 0;
		
		// First, we scan the start point
		scanset = RegularExplorer.explore(originX, originY, originX + step, originY + step, SAMPLESIZE);
		targets = targets + findTargets(scanset);
	
		System.out.println("Scanning:");
	
		// Now, we start scanning in concentric squares around the startpoint
		for(iterations = 1; iterations <= iterationLimit; iterations++)
		{
			// Move the origin for this scan iteration outwards by "step" each cycle
			originX = startx - (iterations * step) - (0.5 * step);
			originY = starty - (iterations * step) - (0.5 * step);
			
			System.out.println("\n Iteration: " + iterations + " - " + String.valueOf(originX) + " , " + String.valueOf(originY));
			System.out.flush();

			double i = originX;
			double j = originY;
			
			// Scan the top part of the square.
			// ( i increasing by step every iteration, j fixed)
			for( i = originX; i < originX + 2*iterations*step + 1; i = i + step)
			{
				scanset = RegularExplorer.explore(i, j, i + step, j + step, SAMPLESIZE);
				targets = targets + findTargets(scanset);
			}
		
			System.out.print("|");
	
			// Scan the right hand limb, moving "downwards"
			// ( i fixed, j increasing by step every iteration)
			for( j = originY; j < originY + 2*iterations*step + 1; j = j + step)
			{
				scanset = RegularExplorer.explore(i, j, i + step, j + step, SAMPLESIZE);
				targets = targets + findTargets(scanset);
			}
			
			System.out.print("|");
	
			// Scan the lower limb, moving "left"
			// ( i decreasing by step every iteration, j fixed)
			for(/* No setup */; i > originX; i = i - step)
			{
				scanset = RegularExplorer.explore(i, j, i + step, j + step, SAMPLESIZE);
				targets = targets + findTargets(scanset);				
			}
			
			System.out.print("|");
	
			for(/* No setup */; j > originY; j = j - step)
			{
				scanset = RegularExplorer.explore(i, j, i + step, j + step, SAMPLESIZE);
				targets = targets + findTargets(scanset);
			}
		}
		
		return targets;
	}
	
	/** This method is responsible for finding potential targets in a given 
	pointset. This simple implementation looks for a surface gradient
	within the maximum and minimum threshold bounds. Every target it finds 
	produces a line of output in the terminal. It returns the total number 
	of potential targets that it has found.
	*/
	public int findTargets(ThreeDFile pointset)
	{
		double[] x = pointset.getX();
		double[] y = pointset.getY();
		double[] z = pointset.getH();
		
		int checkpoint;

		int nearest = 0;
		double nearestDist = 1000000.0;
		
		int targets = 0;
		
		System.out.print(".");
		System.out.flush();

		// We can cheat a little with this method. The features that we are looking for
		// are axis-aligned, so we only have to check the gradients in the X and Y directions,
		// rather than calculate the gradient of the surface at a particular point in 3D
		for(int point = 0; point < x.length; point++)
		{
			// Find its nearest neighbour in X
			for(checkpoint = 0; checkpoint < x.length; checkpoint++)
			{
				// Find the distance between checkpoint and point
				double distance = x[point]-x[checkpoint];
				
				if(distance > nearestDist)
				{
					nearestDist = distance;
					nearest = checkpoint;
				}
			}
			// Calculate the gradient between these two points.
			double gradient = (z[nearest] - z[point])/(x[nearest] - x[point]);
			
			// If the gradient is between the max and min bounds, print out
			// a summary line to stdout.
			if((gradientThresholdMin <= gradient) && (gradient <= gradientThresholdMax))
			{
				System.out.println(String.valueOf(x[point]) + "," + String.valueOf(y[point]) + "," + String.valueOf(z[point])+ "\n");
				targets++;
			}
			
			// Find its nearest neighbour in Y
			for(checkpoint = 0; checkpoint < y.length; checkpoint++)
			{
				// Find the distance between checkpoint and point
				double distance = y[point]-y[checkpoint];
				
				if(distance > nearestDist)
				{
					nearestDist = distance;
					nearest = checkpoint;
				}
			}
			// Calculate the gradient between these two points.
			gradient = (z[nearest] - z[point])/(y[nearest] - y[point]);
			
			// If the gradient is between the max and min bounds, print out
			// a summary line to stdout.
			if((gradientThresholdMin <= gradient) && (gradient <= gradientThresholdMax))
			{
				System.out.println(String.valueOf(x[point]) + "," + String.valueOf(y[point]) + "," + String.valueOf(z[point])+ "\n");
				targets++;
			}
		}

		return targets;
	}
	
	public static void main(String[] args)
	{
		if(args.length != 4)
		{
			System.out.println("Scanner requires four arguments!");
			System.out.println("x - X coordinate of start point");
			System.out.println("y - Y coordinate of start point");
			System.out.println("radius - radius of area to scan");
			System.out.println("step - size of sample area, " + String.valueOf(Scanner.SAMPLESIZE) + " samples per step");
			System.exit(1);
		}
		
		double x 	= Double.parseDouble(args[0]);
		double y 	= Double.parseDouble(args[1]);
		double radius 	= Double.parseDouble(args[2]);
		double step 	= Double.parseDouble(args[3]);
		
		Scanner scanner = new Scanner(x,y , radius, step); // x, y, radius, step
		scanner.setThreshold(-0.001, 0.001); // We are looking for a gradient of zero
		int results = scanner.scan();
		System.out.println(String.valueOf(results) + " potential targets were found");
	}
}
