/*
 * BigProbe.java
 * Created on 5th July 11:40
 */

import filestore1.ThreeDFile;
import randgen.Box;
import java.io.*;
import java.math.*;
/**
 * Class containing helper methods for sampling a surface
 * @author sara
 */
public class BigProbe{
    //the maximum number of points which can be used to
    //sample a Surface (as restricted by the Surface service)
    private static final int MAX_POINTS = 20;
    //The subdirectory in FileStore where all the files
    //are put. This must be unique, e.g. include your name
    private static final String DIR = "BigProbeUserName";
    //URL for the  FileStore service
    private static final String FILESTORE_AD = "http://server5.gs.unina.it:8080/FileStore1/filestore";

    /** Samples a sub-section of a surface using a specified number of
     * points which are generated randomly. The area to be sampled
     * is defined by a bounding box. 
     * @param boundary the bounding box for the sampling
     * @param count the number of sampling points, count <= 
     * MAX_POINTS (the maximum allowed by the Surface serrvice)
     * @param outName the filename for the 3D sample data
     */
    public static void random(Box boundary, int count, String outName,
			      String surfaceURL){
        if(count <= 0 || count > MAX_POINTS){
            System.err.println("BigProbe: count must be <= MAX_POINTS and >0");
            System.exit(1);
        }
        //the filename for the random 2D data
        final String randName = "Rand.xml";

        //delete the input and output files if they exist
        FileStoreWrapper.delete2D(randName,DIR,FILESTORE_AD);
	FileStoreWrapper.delete3D(outName,DIR,FILESTORE_AD);

	//generate the 2D data
        ServiceWrapper.randgen(boundary,count,randName,DIR,FILESTORE_AD);
	//sample the surface
        ServiceWrapper.probe(randName,DIR,outName,DIR,FILESTORE_AD,
			     surfaceURL);
    }


    /** Samples a sub-section of a surface using a specified number of
     * points which are generated on a regular grid. The area to be sampled
     * is defined by a bounding box. 
     * @param boundary the bounding box for the sampling
     * @param count the number of sampling points, count <= 
     * MAX_POINTS (the maximum allowed by the Surface serrvice)
     * @param outName the filename for the 3D sample data
     */
    public static void regular(Box boundary, int count, 
			       String outName, String surfaceURL){
	if(count <= 0 || count > MAX_POINTS) {
	    System.err.println("BigProbe: count must be <= MAX_POINTS and >0");
	    System.exit(1);
	}
	//extract the grid boundaries
	double startX = boundary.getLowerCornerX();
	double limitX = boundary.getUpperCornerX();
	double startY = boundary.getLowerCornerY();
	double limitY = boundary.getUpperCornerY();

	//calculate the grid dimensions 
	double dx = limitX-startX;
	double dy = limitY-startY;

	//calculate the number of points in each direction
	int nX = 
	    (int)Math.floor(Math.sqrt(count*dx/dy));
	int nY = 
	    (int)Math.floor(Math.sqrt(count*dy/dx));

	if(nX == 0 || nY == 0){
	    System.err.println("box dimensions such that one of counts "+
			       " are zero "+nX+" "+nY);
	}

	//calculate the step size between points
	//and the start value for the sequence in X and Y
	double stepX = (limitX-startX)/(nX);
	startX+=stepX/2;

	double stepY = (limitY-startY)/(nY);
	startY+=stepY/2;

	

	//filenames for the temporary sequence and
	//cross-product data
	final String seqNameX = "SeqX.xml";
	final String seqNameY = "SeqY.xml";
	final String xprodName = "XProd.xml";

	//delete the files if they exist
	FileStoreWrapper.delete1D(seqNameX,DIR,FILESTORE_AD);
	FileStoreWrapper.delete1D(seqNameY,DIR,FILESTORE_AD);
	FileStoreWrapper.delete2D(xprodName,DIR,FILESTORE_AD);
	FileStoreWrapper.delete3D(outName,DIR,FILESTORE_AD);

	//generate the sequences and cross-product
	ServiceWrapper.seqgen(startX,stepX,limitX,seqNameX,DIR,FILESTORE_AD);
	ServiceWrapper.seqgen(startY,stepY,limitY,seqNameY,DIR,FILESTORE_AD);
	ServiceWrapper.xprodgen(seqNameX,DIR,seqNameY,DIR,
				xprodName, DIR,FILESTORE_AD);

	//sample the surface
	ServiceWrapper.probe(xprodName,DIR,outName,DIR,FILESTORE_AD,
			     surfaceURL);
    }

    /** Concatenates files containing 3D data. The files which
     * are concatenated are deleted.
     * @param filenames the filenames of the data to be concatenated
     * (which are then deleted)
     * @param cat the filename for the output
     */
    public static void concatenate3D(String filenames[], String cat){
	//delete the output file if it already exists
	FileStoreWrapper.delete3D(cat,DIR,FILESTORE_AD);

	//need to give the sub-directory in FileStore where
	//the files are stored
	String[] dirs = new String[filenames.length];
	for(int i=0; i< dirs.length;i++)
	    dirs[i] = DIR;

	//concatenate the files
	FileStoreWrapper.cat3D(filenames,dirs,cat,DIR,FILESTORE_AD);

	//delete the files which have been concatenated
	for(int i=0; i< dirs.length;i++){
	    FileStoreWrapper.delete3D(filenames[i],DIR,FILESTORE_AD);
	}
    }

    /** Visualises the 3D data by using gnuplot to 
     * generate postscript which is outputted to a file
     * @param filename the 3D data filename
     * @param epsName the postscript filename
     */
    public static void visualise(String filename, String epsName){
	ServiceWrapper.render(filename,DIR,FILESTORE_AD,epsName);
    }
    /** converts the 3D data stored in FileStore from XML to
     * a format that can be inputted to gnuplot. The reformatted
     * data is outputted to a file in the local directory
     * @param filename the 3D data filename
     * @param gnuName the output filename
     */
    public static void convert(String filename, String gnuName){
	ServiceWrapper.reformat(filename,DIR,FILESTORE_AD,gnuName);
    }

    public static ThreeDFile getOutput(String filename){
	return FileStoreWrapper.getThreeDFile(filename,DIR,FILESTORE_AD);
    }
}
